!***********************************************************************

      subroutine type50(xin,yout,par_v,saved_v,iostat)

! ----------------------------------------------------------------------
!
!     TYPE 50 :ZONE ENVELOPE
!              Combines information generated by TYPE 51 Wall models
!
!     December 6, 1984 Cheol Park
!     Last Updated : July 9, 1985 Daniel Clark
!     Upadted to Fortran 90  May 10, 2007 Cheol Park, NIST
!
!     inputs:
!       tia:      Zone air dry-bulb temperature (C)
!       qisw:     Short Wave (visible) radiant internal gain
!                 from lights (kW)
!       qilw:     Long Wave (infrared) radiant internal gain from people
!                 and equipment (kW)
!       tis(i)    Inner wall surface temperatures (C)
!
!     outputs:
!       tmr:      Mean radiant temperature (C)
!       qwall:    Convective heat gain from building surfaces (kW)
!
!     parameters:
!       izn:      Identification number of the zone, [1,6] (-)
!       ns:       Number of surfaces of the zone, [1,10] (-)
!
!     subprograms called:
!       view
!
!     references:
!     [1]  Walton, G.N., "Thermal Analysis Research Program Reference
!                  Manual",
!                  NBSIR 83-2655, National Bureau of Standards,
!                  March 1983.
!     [2]  ASHRAE, 1981 Fundamentals Handbook, ASHRAE, 1981.
!
! **********************************************************************

      use modsim_head
      implicit none

      integer,parameter   :: pp=selected_real_kind(15)
      real(kind=pp)       :: sum,sum1

      integer,parameter   :: ni=maxns+3, no=2, np=2, ns=0
      real,dimension(ni),intent(in)     :: xin
      real,dimension(no),intent(out)    :: yout
      real,dimension(np),intent(in)     :: par_v
      real,dimension(ns),intent(in out) :: saved_v
      integer,dimension(no),intent(out) :: iostat
      real,dimension(maxns)             :: tis
      integer                           :: izn,i,j,ns1
      real                              :: tia,qisw,qilw,qwal,tmr,qwall

      izn    =nint(par_v(1))
      ns1    =nint(par_v(2))

      tia     =xin(1)

!     Convert radiant internal gains from kW to watts.

      qisw    =xin(2)*1000.0
      qilw    =xin(3)*1000.0
      do i=1,ns1
        tis(i)  =xin(i+3)
      end do

!     View angle calculation at the beginning of simulation.

      if(itime==1 ) then
        call view(as,fview,maxns,ns1,maxzn,izn)
      endif

!     Short wave radiant heat fluxes from the sun and the lights

      sum=0.0
      sum1=0.0
      do j=1,ns1
        sum=sum+as(j,izn)*transm(j,izn)*sc(j,izn)*solint(j,izn)
        sum1=sum1+as(j,izn)*(absis(j,izn)+transm(j,izn))
      end do
      if(sum1>0.0) then
        fqrad(izn)=(sum+qisw)/sum1
      else
        fqrad(izn)=0.0
      endif

!     Heat gain/loss from walls

      sum=0.0
      do j=1,ns1
        sum=sum+hisc(j,izn)*as(j,izn)*(tis(j)-tia)
      end do

!     Convert wall convective heat flow from watts to kw.

      qwall=0.001*sum

!     Compute mean radiant temperature of the zone.

      sum=0.0
      sum1=0.0
      do j=1,ns1
        sum=sum+hisr(j,izn)*as(j,izn)*tis(j)
        sum1=sum1+hisr(j,izn)*as(j,izn)
      end do
      if(sum1<=0.0) then
        tmr=tia
      else
        tmr=(sum+qilw)/sum1
      endif

      yout(1)  =tmr
      yout(2)  =qwall

      iostat(1)=0
      iostat(2)=0

      return
      end subroutine type50
! **********************************************************************

      subroutine type51(xin,yout,par_v,saved_v,iostat)

! ----------------------------------------------------------------------
!
!     TYPE 51 : BUILDING SURFACE
!               Compute outer and inner surface temperatures of a
!               surface  construct.
!
!     December 17, 1984 Cheol Park
!     Last updated:  July 9, 1985   Daniel Clark
!     Upadted to Fortran 90  May 10, 2007 Cheol Park, NIST
!
!     inputs:
!       tia:      Indoor air dry-bulb temperature (C)
!       tmr:      Mean radiant temperature (C)
!       tosinf:   Outer surface temperature of unexposed surface (C)
!       fshadw:   Shadowed fraction of exposed outer surface (-)
!
!     outputs:
!       tis:      Inner surface temperature (C)
!       solint:   Average solar flux on outer surface (W/m2)
!
!     parameters:
!       izn:      Identification number of zone, [1,6] (-)
!       id:       Identification number of surface, [1,10] (-)
!       iexpos:   0 = within zone, 1 = between zones, 2 = exposed to
!                 sunlight
!       istr:     Identification number of construct, [1,10] (-)
!       as:       Surface area (m**2)
!       orient:   Angle from the south to the projection of normal to
!                 the surface onto the horizontal plane in clockwise
!                 direction (surface azimuth angle)    (degree)
!       tilt:     Tilt angle of the surface --- the angle between the
!                 normal to the surface and the normal to the horizonal
!                 plane
!       grf:      Ground reflectivity (-)
!       irofs:    Outside surface roughness index, [1,6] (-)
!                 = 1 --- Stucco
!                 = 2 --- Brick, rough plaster
!                 = 3 --- Concrete
!                 = 4 --- Clear pine
!                 = 5 --- Smooth plaster
!                 = 6 --- Glass, paint on pine
!
!       absos:    Solar absorptance of the outer surface (-)
!       absis:    Short wave absorptance of the inner surface (-)
!       emitis:   Emissivity of the inner surface (-)
!       transm:   Transmittance of the glass window (-)
!       sc:       Shading coefficient of the glass window (-)
!
!     saved workspace variables:
!
!       nctf = Number of conduction transfer function terms
!            = Number of past temperature terms used in computing
!                current temperatures and heat fluxes
!             (0 <= nctf <= maxnrf)
!       nord = Order of conduction transfer function calculation
!            = Number of past heat flux terms used in computing
!                current temperatures and heat fluxes
!              (0 <= nord <= maxord)
!              ( nctf = 0  <=>  nord = 0 )
!       tis  = Temperature of inside surface of wall
!       tos  = Temperature of outside surface of wall
!       qi   = Conduction heat flux into wall at inner surface
!       qo   = Conduction heat flux into wall at outer surface
!       qip  = Current value of QI excluding terms at current time
!       qop  = Current value of QO excluding terms at current time
!
!       saved_v(1) = time
!      (saved_v(2+i),i=0,nctf)        = value of tis, i time steps ago
!      (saved_v(3+nctf+i),i=0,nctf)   = value of tos, i time steps ago
!      (saved_v(4+2*nctf+i),i=0,nord) = value of qi, i time steps ago
!      (saved_v(5+2*nctf+nord+i),i=0,nord) = value of qo, i time steps ago
!       saved_v(6+2*(nctf+nord)) = qip
!       saved_v(7+2*(nctf+nord)) = qop
!
!     subprograms called:
!       hiscf
!     references:
!     [1]  Walton, G.N., "Thermal Analysis Research Program Reference
!                 Manual",  Nat'l Bur. of Standards,
!                 NBSIR 83-2655, March 1983.
!     [2]  ASHRAE, 1981 Fundamentals Handbook, ASHRAE, 1981.
!     [3]  Kusuda,T., "NBSLD, the Computer Program for Heating and
!                 Cooling Loads in Buildings," Nat'l Bur. of Standards,
!                 BSS 69, July 1976.
!
! **********************************************************************

      use modsim_head
      implicit none

      integer,parameter   :: pp=selected_real_kind(15)
      real(kind=pp)       :: qip,qop

      integer,parameter   :: ni=4, no=2, np=14,&
                             ns=2*(maxnrf+maxord)+7
      real,dimension(ni),intent(in)     :: xin
      real,dimension(no),intent(out)    :: yout
      real,dimension(np),intent(in)     :: par_v
      real,dimension(ns),intent(in out) :: saved_v
      integer,dimension(no),intent(out) :: iostat
      integer             :: irofs,nord,nordp1,nctf,nctfp1,i,itis,itos,&
                             nplus1,nplus2,j,izn,id,iexpos,istr
      real                :: orient,tilt,grf,absos,emitis,ctilt,csazm,&
                             cst,csalt,ctheta,fsg,fss,hgrd,fsolow,tis,&
                             hos,qsi,f1,tos,aa,bb,qi,qo,tia,tmr,tosinf,&
                             fshadw,x,y,z,hiscf,hsirf,hosf,hisrf
      real                :: dtr=0.0174533

      tia            =xin(1)
      tmr            =xin(2)
      tosinf         =xin(3)
      fshadw         =xin(4)

      izn            =nint(par_v(1))
      id             =nint(par_v(2))
      iexpos         =nint(par_v(3))
      istr           =nint(par_v(4))
      as(id,izn)     =par_v(5)
      orient         =par_v(6)
      tilt           =par_v(7)
      grf            =par_v(8)
      irofs          =nint(par_v(9))
      absos          =par_v(10)
      absis(id,izn)  =par_v(11)
      emitis         =par_v(12)
      transm(id,izn) =par_v(13)
      sc(id,izn)     =par_v(14)

      nord=norda(istr)
      nordp1=nord+1
      nctf=nctfa(istr)
      nctfp1=nctf+1

!    Update saved temperature and heat flux histories on the first
!    call of each time step.

      if(time.ne.saved_v(1)) then
        saved_v(1)=time

!    On first step of simulation without previous initialization,
!    initialize temperature history in saved workspace

        if(itime==1 .and. init==0) then
          if(iexpos<=1) then
            tosinf=toa
          end if
          do i=2,nctfp1
            saved_v(i)=tmr
            saved_v(i+nctfp1)=tosinf
          end do
        endif

        if(time<tshell .and. init/=0) goto 40

!       Begin recalculation of past heat fluxes qip and qop;
!       update saved history of tis and tos

        qip=0.0
        qop=0.0
        if(nctf>0) then
          do i=nctf,1,-1
            itis=i+1
            itos=itis+nctfp1
            qip=qip+yc(i,istr)*saved_v(itos)-zc(i,istr)*saved_v(itis)
            qop=qop+yc(i,istr)*saved_v(itis)-xc(i,istr)*saved_v(itos)
            saved_v(itis+1)=saved_v(itis)
            saved_v(itos+1)=saved_v(itos)
          end do

!         Continue recalculation of qip and qop;
!         update saved history of qi and qo

          nplus1=2*nctfp1+1
          nplus2=nplus1+nordp1
          do j=nord,1,-1
            qip=qip+rj(j,istr)*saved_v(nplus1+j)
            qop=qop+rj(j,istr)*saved_v(nplus2+j)
            saved_v(nplus1+j+1)=saved_v(nplus1+j)
            saved_v(nplus2+j+1)=saved_v(nplus2+j)
          end do
        endif
        saved_v(6+2*(nctf+nord))=qip
        saved_v(7+2*(nctf+nord))=qop
      endif

40    qip=saved_v(6+2*(nctf+nord))
      qop=saved_v(7+2*(nctf+nord))

!     Compute cosines for solar heat gain calculation.

      if(iexpos==2 ) then
        ctilt=cos(tilt*dtr)
        if(ssazm>=1.0) then                       ! 5/10/07
          csazm=0.0
        else  
          csazm=sqrt(1.-ssazm*ssazm)
        endif  
        cst=csazm*cos(orient*dtr)+ssazm*sin(orient*dtr)
        if(ssalt>=1.0) then                       ! 5/10/07
          csalt=0.0
        else  
          csalt=sqrt(1.-ssalt*ssalt)
        endif  
        ctheta=csalt*cst*sin(tilt*dtr)+ssalt*ctilt
        if(ctheta<=0.0) ctheta=0.0

!       Angle factors between ground and surface and between sky and
!       surface

        fsg=0.5*(1.-ctilt)
        fss=0.5*(1.+ctilt)

!       Diffusive radiation intensity from ground

        hgrd=hhor*grf

!       Determine the solar heat flux on the exposed surface.

        solint(id,izn)=hdn*ctheta*(1-fshadw)+hsky*fss+hgrd*fsg
        fsolow=absos*solint(id,izn)
      else
        solint(id,izn)=0.0
        fsolow=0.0
      endif

!     Calculate heat transfer coefficients

      if (itime == 1) then                    ! added 7/26/2001
         tis = 20.0
      endif

      hisc(id,izn)=hiscf(tis,tia,tilt)
      if(itime==1) then
        hisr(id,izn)=hisc(id,izn)
      else
        hisr(id,izn)=hisrf(tis,fview(id,izn),emitis)
      endif
      if(iexpos==2) then
        hos=hosf(vwind,irofs)
      else
        hos=0.0
      endif

!     Surface temperatures are obtained from the surface heat
!     balance conditions.

      qsi=absis(id,izn)*fqrad(izn)

!     Outside surface exposed to sunlight

      if(iexpos==2) then
        f1=yc(0,istr)/(zc(0,istr)+hisc(id,izn)+hisr(id,izn))
        tos=(qop+fsolow+hos*toa+f1*(qip+qsi+hisc(id,izn)*tia+&
             hisr(id,izn)*tmr))/(xc(0,istr)+hos-f1*yc(0,istr))

!     Outside surface exposed to another zone or to ground:
!     outside surface temperature is an input

      elseif(iexpos==1) then
        tos=tosinf

!     Within zone: outside surface temp. equals inside surface temp.

      else
        tos=(hisc(id,izn)*tia+hisr(id,izn)*tmr+qip+qsi)/(hisc(id,izn)+&
             hisr(id,izn)+zc(0,istr)-yc(0,istr))
      endif

      if(iexpos>=1) then
        aa=hisc(id,izn)*tia+hisr(id,izn)*tmr+qip+qsi+yc(0,istr)*tos
        bb=hisc(id,izn)+hisr(id,izn)+zc(0,istr)
        tis=aa/bb
      else
        tis=tos
      endif

!     Calculate current heat fluxes at inside and outside surfaces

      qi=yc(0,istr)*tos-zc(0,istr)*tis+qip
      qo=yc(0,istr)*tis-xc(0,istr)*tos+qop

!     Save current temperatures and heat fluxes

      saved_v(2)=tis
      saved_v(2+nctfp1)=tos
      saved_v(2+nctfp1+nctfp1)=qi
      saved_v(2+nctfp1+nctfp1+nordp1)=qo

!     Set outputs

      yout(1)=tis
      yout(2)=solint(id,izn)

      iostat(1)=0
      iostat(2)=0

      return
      end subroutine type51

! **********************************************************************

      subroutine type52(xin,yout,par_v,saved_v,iostat)

! ----------------------------------------------------------------------
!
!     TYPE 52 : ZONE MODEL
!               Compute zone air dry-bulb temperture and humidity ratio.
!
!     December 26, 1984 Cheol Park
!     Last Updated : July 9, 1985  Daniel Clark
!     Upadted to Fortran 90  May 10, 2007 Cheol Park, NIST
!
!     inputs:
!       piag:     Gage pressure of zone air (kPa)
!       tia:      Zone air dry-bulb temperature (C)
!       wia:      Humidity ratio of zone air (-)
!       psag:     Gage pressure of supply air (kPa)
!       flwsa:    Mass flow rate of supply air (kg/s)
!       tsa:      Supply air dry-bulb temperature (C)
!       wsa:      Humidity ratio of supply air (-)
!       qwall:    Convective heat gain from building surfaces (kW)
!       numpep:   Number of persons in the zone (-)
!       utceqp:   Equipment utilization coefficient, [0,1] (-)
!       utclit:   Lighting utilization coefficient, [0,1] (-)
!
!
!     outputs:
!       dtidt:    Derivative of TIA with respect to time (C/s)
!                 Use the same index number as TIA.
!       dwidt:    Derivative of WIA with respect to time (1/s)
!                 Use the same index number as WIA.
!       qisw:     Short Wave (visible) radiant internal gain
!                 from lights (kW)
!       qilw:     Long Wave (thermal) radiant internal gain from people,
!                 equipment, and lights (kW)
!
!     parameters:
!       izn:      Identification number of zone (-)
!       cfur:     Effective thermal capacitance of furnishing (kJ/K)
!       effmia:   Air mass multiplier for moisture capacitance of
!                 zone (-)
!       volume:   Volume of zone air (interior space of zone) (m**3)
!       sairex:   Standard air exchange (infiltration) rate (1/hr)
!                  Living space --  1.5 for leaky building
!                                   1.0 for standard building
!                                   0.5 for modern-type building
!                  Attic space  -- 20.0 for mechanical ventilation
!                                   6.0 for natural ventilation
!                  Crawl space  --  3.0
!
!       wpeps:    Sensible heat gain from a person (kW)
!       wpepl:    Latent heat gain from a person (kW)
!       wlit:     Heat gain due to lighting in the zone (kW)
!       light:    Type of lighting: 1=>fluorescent, 2=>incandescent
!       weqps:    Sensible heat gain due to equipment (kW)
!       weqpl:    Latent heat gain due to equipment (kW)
!       reqp:     Ratio of radiative heat to total sensible heat from
!                 equipment,[0,1] (-)
!
!     subprograms called:
!       cp,densit,airex,hfg
!
! **********************************************************************

      use modsim_head
      implicit none

      integer,parameter   :: ndeq=2, ni=11, no=4, np=12, ns=0
      real,dimension(ni),intent(in)     :: xin
      real,dimension(no),intent(out)    :: yout
      real,dimension(np),intent(in)     :: par_v
      real,dimension(ns),intent(in out) :: saved_v
      integer,dimension(no),intent(out) :: iostat
      integer             :: izn,light,numpep
      real                :: cfur,effmia,volume,sairex,wpeps ,wpepl,&
                             wlit,weqps,weqpl,reqp,qpepc,qeqpc,&
                             qlitc,qeqpl,qpepl,qlat,qpepr,qeqpr,&
                             qlitr,qilw,qisw,pwoa,pia,pwia,deninf,&
                             flwinf,qinf,dens,cair,qsa,dtidt,wsat,&
                             effm,dwidt,piag,tia,wia,psag,flwsa,tsa,&
                             wsa,qwall,utceqp,utclit,pwf,&
                             densit,airex,cp,wsatf,hfg
      real,dimension(2)   :: fshort=(/0.2,0.1/),flong= (/0.2,0.8/),&
                             fconv= (/0.6,0.1/)
      real                :: rpep=0.7

      piag   =xin(1)
      tia    =xin(2)     
      wia    =xin(3)     
      psag   =xin(4)     
      flwsa  =xin(5)     
      tsa    =xin(6)
      wsa    =xin(7)
      qwall  =xin(8)

!     The following input values are boundary values, which may reside
!     in the boundary data file.

      numpep  =nint(xin(9))
      utceqp  =xin(10)
      utclit  =xin(11)

      izn    =nint(par_v(1))
      cfur   =par_v(2)
      effmia =par_v(3)
      volume =par_v(4)
      sairex =par_v(5)
      wpeps  =par_v(6)
      wpepl  =par_v(7)
      wlit   =par_v(8)
      light  =nint(par_v(9))
      weqps  =par_v(10)
      weqpl  =par_v(11)
      reqp   =par_v(12)

!     Convective heat gain from people occupying the zone

      qpepc=(1-rpep)*numpep*wpeps

!     Convective heat gain from equipment

      qeqpc=(1-reqp)*utceqp*weqps

!     Convective heat gain from lighting

      qlitc=fconv(light)*utclit*wlit

!     Latent heat gain from equipment and people

      qeqpl=weqpl*utceqp
      qpepl=numpep*wpepl
      qlat=qpepl+qeqpl

!     Long wave radiant heat gains from people, lights, and equipment

      qpepr=rpep*numpep*wpeps
      qeqpr=reqp*utceqp*weqps
      qlitr=flong(light)*utclit*wlit
      qilw=qpepr+qeqpr+qlitr

!     Short wave radiant heat gains from lights

      qisw=fshort(light)*utclit*wlit

!     Sensible heat gain or loss due to infiltration

      pwoa=pwf(woa,poa)
      pia=piag+poa
      pwia=pwf(wia,pia)
      deninf=0.5*(densit(poa,toa,woa,pwoa)+densit(pia,tia,wia,pwia))
      flwinf=deninf*volume*airex(sairex,vwind,toa,tia)/3600.0
      qinf=flwinf*(cp(woa)*toa-cp(wia)*tia)

!     Zone air temperature is to be determined from the zone air heat
!     balance equation ( 1st order differential equation ).

      pia=piag+poa
      pwia=pwf(wia,pia)
      dens=densit(pia,tia,wia,pwia)
      cair=cp(wia)*dens*volume
      qsa=cp(wsa)*flwsa*(tsa-tia)
      dtidt=(qsa+qwall+qinf+qlitc+qpepc+qeqpc)/(cfur+cair)

!     Zone air humidity ratio is computed from the zone air moisture
!     balance equation ( 1st order differential equation ).

      wsat=wsatf(tia,pia)
      if(wia>wsat) then
        wia=wsat
      end if
      effm=dens*volume*effmia
      dwidt=(qlat/hfg(tia)+flwinf*(woa-wia)+flwsa*(wsa-wia))/effm

      yout(1)=dtidt
      yout(2)=dwidt
      yout(3)=qisw
      yout(4)=qilw
      iostat(1)=0
      iostat(2)=0
      iostat(3)=0
      iostat(4)=0

      return
      end subroutine type52

!***********************************************************************

      subroutine type53(xin,yout,par_v,saved_v,iostat)

!-----------------------------------------------------------------------
!
!     TYPE 53 : WEATHER INPUT
!
!      To place weather data read by subroutine RDENV into the
!      state vector.  The inputs are really just for mnemonic
!      purposes.  The parameters are the indices of the variables.
!      input indices should always equal parameter values.
!
!     Use:  Does nothing when building shell model is not used.
!      Optional when building shell model is used.
!      One unit per simulation is sufficient.
!      Note that weather variables are already available,
!      interpolated separately for each superblock. This TYPE routine is
!      needed only when weather variables are to be used as inputs or
!      reported variables.
!
!     inputs:
!       tamb:   Ambient (outdoor) air temperature (C)
!       humrat: Outdoor air humidity ratio (-)
!       pbar:   Barometric pressure (kPa)
!       idn:    Direct normal solar radiation (W/m*m)
!       hsky:   Diffuse (sky) solar radiation (W/m*m)
!       ihor:   Total horizontal solar radiation (W/m*m)
!
!     outputs:
!       none
!
!     parameters:
!       ntoa:   Index for ambient temperature (e.g. 5 if tamb = t5)
!       nwoa:   Index for outdoor air humidity ratio
!       npoa:   Index for barometric pressure
!       ndn:    Index for direct normal solar radiation
!       nsky:   Index for diffuse (sky) solar radiation
!       nhor:   Index for total horizontal solar radiation
!
!     example:  Suppose:
!       1) Outdoor ambient temperature and humidity ratio are used
!          as input state variables t5 and h3 somewhere in a simulation.
!       2) total horizontal solar radiation is not an input but is to be
!          reported under the label q2.
!       3) The remaining weather variables are not needed as state
!          variables.
!
!               Inputs:      5, 3, 0, 0, 0, 2
!               Parameters:  5, 3, 0, 0, 0, 2
!
!***********************************************************************

      use modsim_head
      implicit none

      integer,parameter   :: ni=6, no=6, np=6, ns=0
      real,dimension(ni),intent(in)     :: xin
      real,dimension(no)                :: yout
      real,dimension(np),intent(in)     :: par_v
      real,dimension(ns)                :: saved_v
      integer,dimension(no)             :: iostat
      real,dimension(ni)                :: dummy
      integer                           :: i

!     Inputs

      do i=1,ni
        dummy(i) = xin(i)
      end do

!     Parameters

      ntoa  =nint(par_v(1))
      nwoa  =nint(par_v(2))
      npoa  =nint(par_v(3))
      ndn   =nint(par_v(4))
      nsky  =nint(par_v(5))
      nhor  =nint(par_v(6))

      return
      end subroutine type53

!***********************************************************************

      subroutine type54(xin,yout,par_v,saved_v,iostat)

! ----------------------------------------------------------------------
!
!     TYPE 54 :ZONE ENVELOPE
!              Combines information generated by TYPE 51 Wall models
!
!     December 6, 1984 Cheol Park
!     Last Updated : July 9, 1985 Daniel Clark
!     Upadted to Fortran 90  May 10, 2007 Cheol Park, NIST
!
!     inputs:
!       tia:      Zone air dry-bulb temperature (C)
!       tis(i)    Inner wall surface temperatures (C)
!       qisw:     Short Wave (visible) radiant internal gain
!                 from lights (kW)
!       qilw:     Long Wave (infrared) radiant internal gain from people
!                 and equipment (kW)
!
!     outputs:
!       tmr:      Mean radiant temperature (C)
!       qwall:    Convective heat gain from building surfaces (kW)
!
!     parameters:
!       izn:      Identification number of the zone, [1,6] (-)
!       ns:       Number of surfaces of the zone, [1,10] (-)
!
!     subprograms called:
!       view
!
!     references:
!     [1]  Walton, G.N., "Thermal Analysis Research Program Reference
!                  Manual",
!                  NBSIR 83-2655, National Bureau of Standards,
!                  March 1983.
!     [2]  ASHRAE, 1981 Fundamentals Handbook, ASHRAE, 1981.
!
! **********************************************************************

      use modsim_head
      implicit none

      integer,parameter   :: pp=selected_real_kind(15)
      real(kind=pp)       :: sum,sum1

      integer,parameter   :: ni=maxns+3, no=2, np=2, ns=0
      real,dimension(ni),intent(in)     :: xin
      real,dimension(no),intent(out)    :: yout
      real,dimension(np),intent(in)     :: par_v
      real,dimension(ns),intent(in out) :: saved_v
      integer,dimension(no),intent(out) :: iostat
      real,dimension(maxns)             :: tis
      integer                           :: i,j,izn,ns1
      real                              :: tia,qisw,qilw,qwall,tmr

      izn  =nint(par_v(1))
      ns1  =nint(par_v(2))

      tia  =xin(1)

!     Convert radiant internal gains from kw to watts.

      qisw =xin(2)*1000.0
      qilw =xin(3)*1000.0
      do i=1,ns1
        tis(i)  =xin(i+3)
      end do

!     View angle calculation at the beginning of simulation.

      if(itime==1 ) then
        call view(as,fview,maxns,ns1,maxzn,izn)
      endif

!     Short wave radiant heat fluxes from the sun and the lights

      sum=0.0
      sum1=0.0
      do j=1,ns1
        sum=sum+as(j,izn)*transm(j,izn)*sc(j,izn)*solint(j,izn)
        sum1=sum1+as(j,izn)*(absis(j,izn)+transm(j,izn))
      end do
      if(sum1>0.0) then
        fqrad(izn)=(sum+qisw)/sum1
      else
        fqrad(izn)=0.0
      endif

!     Heat gain/loss from walls

      sum=0.0
      do j=1,ns1
        sum=sum+hisc(j,izn)*as(j,izn)*(tis(j)-tia)
      end do

!     Convert wall convective heat flow from watts to kw.

      qwall=0.001*sum

!     Compute mean radiant temperature of the zone.

      sum=0.0
      sum1=0.0
      do j=1,ns1
        sum=sum+hisr(j,izn)*as(j,izn)*tis(j)
        sum1=sum1+hisr(j,izn)*as(j,izn)
      end do
      if(sum1<=0.0) then
        tmr=tia
      else
        tmr=(sum+qilw)/sum1
      endif

      yout(1)  =tmr
      yout(2)  =qwall

      iostat(1)=0
      iostat(2)=0

      return
      end subroutine type54

